/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Reflection;
using System.Windows.Forms;
using System.ComponentModel;

using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Handles;

namespace Borland.Eco.WinForm
{
	public sealed class DraggedElement
	{
		private readonly IElement m_Element;
		public DraggedElement(IElement element)
		{
			m_Element = element;
		}

		public IElement Element
		{
			get { return m_Element; }
		}
	}

	///<summary>
	///A collection of useful helper functions pertaining to Windows.Forms.
	///</summary>
	public sealed class EcoWinFormHelper
	{
		private EcoWinFormHelper() {}
		public const string PropertyNameDataSource = "DataSource";
		public const string PropertyNameReadOnly = "ReadOnly";

		///<exception cref="ArgumentNullException">Thrown if <paramref name="e"/> is null.</exception>
		public static bool ContainsDroppable(System.Windows.Forms.DragEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e"); // Do not localize
			return e.Data.GetDataPresent(typeof(DraggedElement));
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="e"/> is null.</exception>
		public static IElement GetDroppedElement(System.Windows.Forms.DragEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e"); // Do not localize
			return ((DraggedElement)e.Data.GetData(typeof(DraggedElement))).Element;
		}
		///<summary>
		///Returns the value of the property DataSource or null if not applicable.
		///</summary>
		///<returns>Returns null if there is no property named DataSource - or the value of that property.</returns>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public static object ConnectedDataSource(object o)
		{
			if (o == null) throw new ArgumentNullException("o"); // Do not localize
			PropertyInfo pi = o.GetType().GetProperty(PropertyNameDataSource);
			if (pi == null) return null;
			return pi.GetValue(o, null);
		}			
		
		///<summary>
		///Returns the datasource soft cast to an ElementHandle.
		///</summary>
		public static ElementHandle ConnectedHandle(object o)
		{
			return ConnectedDataSource(o) as ElementHandle;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public static int GetCurrentIndex(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			Object dataSource = ConnectedDataSource(control);

			if ((dataSource == null) && (control.DataBindings.Count > 0))
				dataSource = control.DataBindings[0].DataSource;

			return control.BindingContext[dataSource].Position;
		}
		public static IElementCollection GetElementCollectionFromDataSource(object sender)
		{
			ElementHandle elementHandle = ConnectedHandle(sender);
			if (elementHandle != null)
			{
				// Extract the object list
				IElement el = elementHandle.Element;
				if (el != null)
					return el.GetAsCollection();
			}
			return null;
		}

		///<summary>
		///Returns the value of the boolean property ReadOnly. Default is false.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="sender"/> is null.</exception>
		public static bool GetIsReadOnly(object sender)
		{
			if (sender == null) throw new ArgumentNullException("sender"); // Do not localize
			bool res = false;
			PropertyInfo pi = sender.GetType().GetProperty(PropertyNameReadOnly);
			if (pi != null)
			{
				Object prop = pi.GetValue(sender, null);
				if (prop is System.Boolean)
					res = (bool)prop;
			}
			return res;
		}

		///<summary>
		///Returns the EcoSpace the elementHandle is ultimately connected to - or null if the 
		///handle is not connected to an EcoSpace.
		///</summary>
		public static EcoSpace ConnectedEcoSpace(ElementHandle elementHandle)
		{
			if (elementHandle == null) return null;
			if (elementHandle is RootedHandle)
				return ConnectedEcoSpace(((RootedHandle)elementHandle).RootHandle);
			if (elementHandle is RootHandle)
				return ((RootHandle)elementHandle).EcoSpace;
			return null;
		}
		///<summary>
		///Sets the Enabled-property on a component if the component is a Control or a MenuItem.
		///Helper function for extenders.
		///</summary>
		public static void SetEnabledOnComponent(Component component, bool enabled)
		{
			Control control = component as Control;
			if (control != null)
				control.Enabled = enabled;
			else
			{
				MenuItem menuItem = component as MenuItem;
				if (menuItem != null)
					menuItem.Enabled = enabled;
			}
		}

		///<summary>
		///Sets the caption on a component if the component is a Control or a MenuItem.
		///Helper function for extenders.
		///</summary>
		public static void SetCaptionOnComponent(Component component, string caption)
		{
			Control control = component as Control;
			if (control != null)
				control.Text = caption;
			else
			{
				MenuItem menuItem = component as MenuItem;
				if (menuItem != null)
					menuItem.Text = caption;
			}
		}

		///<summary>
		///Gets the caption on a component if the component is a Control or a MenuItem.
		///Helper function for extenders.
		///</summary>
		public static string CaptionFromComponent(Component component)
		{
			Control control = component as Control;
			if (control != null)
				return control.Text;
			MenuItem menuItem = component as MenuItem;
			if (menuItem != null)
				return menuItem.Text;
			return string.Empty;
		}
	}
}
